/* Copyright (C) 2003 Vladimir Roubtsov. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 * 
 * $Id: Property.java,v 1.1.1.1 2004/05/09 16:57:54 vlad_r Exp $
 */
package com.vladium.util;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

// ----------------------------------------------------------------------------
/**
 * @author Vlad Roubtsov, (C) 2003
 */
public
abstract class Property
{
    // public: ................................................................


    public static boolean toBoolean (final String value)
    {
        if (value == null)
            return false;
        else
            return value.startsWith ("t") || value.startsWith ("y");
    }

    /**
     * @param properties [null is equivalent to an empty set]
     * @param prefix [may not be null]
     */
    public static String [] toAppArgsForm (final Properties properties, final String prefix)
    {
        if (properties == null)
            return IConstants.EMPTY_STRING_ARRAY;
        else
        {
            if (prefix == null) throw new IllegalArgumentException ("null input: prefix");
            
            final List _result = new ArrayList (properties.size ());
            for (Enumeration names = properties.propertyNames (); names.hasMoreElements (); )
            {
                final String name = (String) names.nextElement ();
                final String value = properties.getProperty (name, "");
                
                _result .add (prefix.concat (name).concat ("=").concat (value)); 
            }
            
            if (_result.isEmpty ())
                return IConstants.EMPTY_STRING_ARRAY;
            else
            {                
                final String [] result = new String [_result.size ()];
                _result.toArray (result);
                
                return result;
            }
        }
    }    
    
    /**
     * @param overrides [null is equivalent to empty]
     * @param base [null is equivalent to empty]
     * 
     * @return [never null, could be empty]
     */
    public static Properties combine (final Properties overrides, final Properties base)
    {
        // note: no defensive copies here
         
        if (base == null)
        {
            if (overrides == null)
                return new XProperties ();
            else
                return overrides;
        }
        
        // [assertion: base != null]
        
        if (overrides == null) return base;
        
        // [assertion: both 'overrides' and 'base' are not null]
        
        final Properties result = new XProperties (base);
        
        // note: must use propertyNames() because that is the only method that recurses
        // into possible bases inside 'overrides'
        
        for (Enumeration overrideNames = overrides.propertyNames (); overrideNames.hasMoreElements (); )
        {
            final String n = (String) overrideNames.nextElement ();
            final String v = overrides.getProperty (n);
            
            result.setProperty (n, v);
        }
        
        return result;
    }
    
    public static Properties getSystemProperties (final String systemPrefix)
    {
        // note: this method is not synchronized on purpose
        
        Properties result = s_systemProperties;
        if (result == null)
        {
            result = new SystemPropertyLookup (systemPrefix);
            
            s_systemProperties = result;
            return result;
        }
        
        return result;
    }
    
    public static Properties getSystemPropertyRedirects (final Map systemRedirects)
    {
        // note: this method is not synchronized on purpose
        
        Properties result = s_systemRedirects;
        if (result == null)
        {
            result = new SystemRedirectsLookup (systemRedirects);
            
            s_systemRedirects = result;
            return result;
        }
        
        return result;
    }


    public static String getSystemFingerprint ()
    {
        // [not synchronized intentionally]
        
        if (s_systemFingerprint != null)
            return s_systemFingerprint;
        else
        {
            final StringBuffer s = new StringBuffer ();
            final char delimiter = ':';
            
            s.append (getSystemProperty ("java.vm.name", ""));
            s.append (delimiter);
            s.append (getSystemProperty ("java.vm.version", ""));
            s.append (delimiter);
            s.append (getSystemProperty ("java.vm.vendor", ""));
            s.append (delimiter);
            s.append (getSystemProperty ("os.name", ""));
            s.append (delimiter);
            s.append (getSystemProperty ("os.version", ""));
            s.append (delimiter);
            s.append (getSystemProperty ("os.arch", ""));
            
            s_systemFingerprint = s.toString ();
            return s_systemFingerprint;
        }
    }    
    
    public static String getSystemProperty (final String key)
    {
        try
        {
            return System.getProperty (key);
        }
        catch (SecurityException se)
        {
            return null;
        }
    }
    
    public static String getSystemProperty (final String key, final String def)
    {
        try
        {
            return System.getProperty (key, def);
        }
        catch (SecurityException se)
        {
            return def;
        }
    }
    
    /**
     * does not throw
     * 
     * @param name
     * @return
     */
    public static Properties getProperties (final String name)
    {
        Properties result = null;
        
        InputStream in = null;
        try
        {
            in = ResourceLoader.getResourceAsStream (name);
            if (in != null)
            {
                result = new XProperties ();
                result.load (in);
            }
        }
        catch (Throwable t)
        {
            result = null;
        }
        finally
        {
            if (in != null) try { in.close (); } catch (Throwable ignore) {}
            in = null;
        }
        
        return result;
    }
    
    /**
     * does not throw
     * 
     * @param name
     * @param loader
     * @return
     */
    public static Properties getProperties (final String name, final ClassLoader loader)
    {
        Properties result = null;
        
        InputStream in = null;
        try
        {
            in = ResourceLoader.getResourceAsStream (name, loader);
            if (in != null)
            {
                result = new XProperties ();
                result.load (in);
            }
        }
        catch (Throwable t)
        {
            result = null;
        }
        finally
        {
            if (in != null) try { in.close (); } catch (Throwable ignore) {}
            in = null;
        }
        
        return result;
    }
    
    
    public static Properties getPropertiesFromFile (final File file)
        throws IOException
    {
        Properties result = null;
        
        InputStream in = null;
        try
        {
            in = new BufferedInputStream (new FileInputStream (file), 8 * 1024);

            result = new XProperties ();
            result.load (in);
        }
        finally
        {
            if (in != null) try { in.close (); } catch (Throwable ignore) {}
            in = null;
        }
        
        return result;
    }    
    
    // protected: .............................................................

    // package: ...............................................................
    
    // private: ...............................................................
    
    
    private static final class SystemPropertyLookup extends XProperties
    {
        // note: due to incredibly stupid coding in java.util.Properties
        // (getProperty() uses a non-virtual call to get(), while propertyNames()
        // uses a virtual call to the same instead of delegating to getProperty())
        // I must override both methods below
        
        public String getProperty (final String key)
        {
            return (String) get (key);
        }
            
        public Object get (final Object key)
        {
            if (! (key instanceof String)) return null;
            
            String result = (String) super.get (key);
            if (result != null) return result;
            
            if (m_systemPrefix != null)
            {
                result = getSystemProperty (m_systemPrefix.concat ((String) key), null);
                
                if (result != null) return result;
            }
            
            return result;
        }
        
        /*
         * Overrides Properties.propertyNames() [this is used for debug logging only]
         */
        public synchronized Enumeration keys ()
        {
            final Hashtable _propertyNames = new Hashtable ();
            
            if (m_systemPrefix != null)
            {
                try
                {
                    final int systemPrefixLength = m_systemPrefix.length ();
                    
                    for (Enumeration e = System.getProperties ().propertyNames ();
                         e.hasMoreElements (); )
                    {
                        final String n = (String) e.nextElement ();
                        
                        if (n.startsWith (m_systemPrefix))
                        {
                            final String yn = n.substring (systemPrefixLength);
                            
                            _propertyNames.put (yn, yn);
                        }
                    } 
                }
                catch (SecurityException ignore)
                {
                    ignore.printStackTrace (System.out);
                    
                    // continue
                }
            }
            
            return _propertyNames.keys ();
        }

                
        SystemPropertyLookup (String systemPrefix)
        {
            if ((systemPrefix != null) && ! systemPrefix.endsWith ("."))
                systemPrefix = systemPrefix.concat (".");
                
            m_systemPrefix = systemPrefix;
        }
        
        
        private final String m_systemPrefix; // can be null [if not null, normalized to end with "."]
        
    } // end of nested class
    
    
    private static final class SystemRedirectsLookup extends XProperties
    {
        // note: due to incredibly stupid coding in java.util.Properties
        // (getProperty() uses a non-virtual call to get(), while propertyNames()
        // uses a virtual call to the same instead of delegating to getProperty())
        // I must override both methods below
        
        public String getProperty (final String key)
        {
            return (String) get (key);
        }
            
        public Object get (final Object key)
        {
            if (! (key instanceof String)) return null;
            
            String result = (String) super.get (key);
            if (result != null) return result;
            
            if (m_systemRedirects != null)
            {
                final String redirect = (String) m_systemRedirects.get (key);
                
                if (redirect != null)
                {
                    result = getSystemProperty (redirect, null);
                    if (result != null) return result;
                }
            }
            
            return result;
        }
        
        /*
         * Overrides Properties.propertyNames() [this is used for debug logging only]
         */
        public synchronized Enumeration keys ()
        {
            final Hashtable _propertyNames = new Hashtable ();
            
            if (m_systemRedirects != null)
            {
                for (Iterator i = m_systemRedirects.keySet ().iterator ();
                     i.hasNext (); )
                {
                    final Object key = i.next ();                    
                    if (key != null) _propertyNames.put (key , key);
                }
            }
            
            return _propertyNames.keys ();
        }

                
        SystemRedirectsLookup (final Map systemRedirects)
        {
            m_systemRedirects = systemRedirects; // note: no defensive copy
        }
        
        
        private final Map m_systemRedirects; // can be null
        
    } // end of nested class
    
    
    private static String s_systemFingerprint;
    private static Properties s_systemProperties, s_systemRedirects;
    
} // end of class
// ----------------------------------------------------------------------------