/* Copyright (C) 2003 Vladimir Roubtsov. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 * 
 * $Id: Logger.java,v 1.1.1.1 2004/05/09 16:57:51 vlad_r Exp $
 */
package com.vladium.logging;

import java.io.PrintWriter;
import java.util.Set;

import com.vladium.util.ClassLoaderResolver;

// ----------------------------------------------------------------------------
/**
 * A simple Java version-independent logging framework. It is configured statically
 * [this should be done before the first call to any method in this class] via
 * {@link LoggerInit} setter methods.<P>
 * 
 * Every log message is structured as follows:
 * <OL>
 *  <LI> message is prefixed with the what was set as {@link LoggerInit#setPrefix}
 * if that is not null;
 *  <LI> if the calling class could be identified and it supplied the calling
 * method name, the calling method is identified with all name components that
 * are not null;
 *  <LI> caller-supplied message is logged, if not null;
 *  <LI> caller-supplied Throwable is dumped starting with a new line, if not null.
 * </OL>
 * 
 * @see ILogLevels
 * 
 * @author (C) 2001, Vlad Roubtsov
 */
public
abstract class Logger implements ILogLevels
{
    // public: ................................................................
    
    // TODO: update javadoc for 'logCaller'
    
    // TODO: need isLoggable (Class)
    
    public static PrintWriter getWriter ()
    {
        return OUT;
    }
    
    /**
     * A quick method to determine if logging is enabled at a given level.
     * This method acquires no monitors and should be used when calling one of
     * log() or convenience logging methods directly incurs significant
     * parameter construction overhead.
     * 
     * @see ILogLevels
     */
    public static boolean isLoggable (final int level)
    {
        return (level <= LEVEL);
    }

    /**
     * A convenience method equivalent to isLoggable(VERBOSE).
     */
    public static boolean atVERBOSE ()
    {
        return (VERBOSE <= LEVEL);
    }
    
    /**
     * A convenience method equivalent to isLoggable(TRACE1).
     */
    public static boolean atTRACE1 ()
    {
        return (TRACE1 <= LEVEL);
    }

    /**
     * A convenience method equivalent to isLoggable(TRACE2).
     */
    public static boolean atTRACE2 ()
    {
        return (TRACE2 <= LEVEL);
    }

    /**
     * A convenience method equivalent to isLoggable(TRACE3).
     */
    public static boolean atTRACE3 ()
    {
        return (TRACE3 <= LEVEL);
    }


    /**
     * A convenience method to log 'msg' from an anonymous calling method
     * at WARNING level.
     * 
     * @param msg log message [ignored if null]
     */
    public static void warning (final String msg)
    {
        _log (WARNING, null, msg, false);
    }
    
    /**
     * A convenience method to log 'msg' from an anonymous calling method
     * at INFO level.
     * 
     * @param msg log message [ignored if null]
     */
    public static void info (final String msg)
    {
        _log (INFO, null, msg, false);
    }
    
    /**
     * A convenience method to log 'msg' from a given calling method
     * at INFO level.
     * 
     * @param method calling method name [ignored if null]
     * @param msg log message [ignored if null]
     */
    public static void info (final String method, final String msg)
    {
        _log (INFO, method, msg, false);
    }
    
    /**
     * A convenience method to log 'msg' from an anonymous calling method
     * at VERBOSE level.
     * 
     * @param msg log message [ignored if null]
     */
    public static void verbose (final String msg)
    {
        _log (VERBOSE, null, msg, false);
    }

    
    /**
     * A convenience method equivalent to log(TRACE1, method, msg).
     * 
     * @param method calling method name [ignored if null]
     * @param msg log message [ignored if null]
     */
    public static void trace1 (final String method, final String msg)
    {
        _log (TRACE1, method, msg, true);
    }
    
    /**
     * A convenience method equivalent to log(TRACE2, method, msg).
     * 
     * @param method calling method name [ignored if null]
     * @param msg log message [ignored if null]
     */
    public static void trace2 (final String method, final String msg)
    {
        _log (TRACE2, method, msg, true);
    }
    
    /**
     * A convenience method equivalent to log(TRACE3, method, msg).
     * 
     * @param method calling method name [ignored if null]
     * @param msg log message [ignored if null]
     */
    public static void trace3 (final String method, final String msg)
    {
        _log (TRACE3, method, msg, true);
    }

    /**
     * Logs 'msg' from an unnamed calling method.
     *  
     * @param level level to log at [the method does nothing if this is less
     * than the set level].
     * @param msg log message [ignored if null]
     */        
    public static void log (final int level, final String msg, final boolean logCaller)
    {
        _log (level, null, msg, logCaller);
    }
    
    /**
     * Logs 'msg' from a given calling method.
     *  
     * @param level level to log at [the method does nothing if this is less
     * than the set level].
     * @param method calling method name [ignored if null]
     * @param msg log message [ignored if null]
     */        
    public static void log (final int level, final String method, final String msg, final boolean logCaller)
    {
        _log (level, method, msg, logCaller);
    }

    /**
     * Logs 'msg' from an unnamed calling method followed by the 'throwable' stack
     * trace dump.
     *  
     * @param level level to log at [the method does nothing if this is less
     * than the set level].
     * @param msg log message [ignored if null]
     * @param throwable to dump after message [ignored if null]
     */
    public static void log (final int level, final String msg, final Throwable throwable)
    {
        _log (level, null, msg, throwable);
    }
    
    /**
     * Logs 'msg' from a given calling method followed by the 'throwable' stack
     * trace dump.
     *  
     * @param level level to log at [the method does nothing if this is less
     * than the set level].
     * @param method calling method name [ignored if null]
     * @param msg log message [ignored if null]
     * @param throwable to dump after message [ignored if null]
     */
    public static void log (final int level, final String method, final String msg, final Throwable throwable)
    {
        _log (level, method, msg, throwable);
    }

    // protected: .............................................................

    // package: ...............................................................

    // private: ...............................................................
    
    
    private Logger () {} // prevent subclassing
    
    
    private static void _log (final int level, final String method,
                              final String msg, final boolean logCaller)
    {                
        if ((level <= LEVEL) && (level >= SEVERE))
        {
            final PrintWriter out = OUT; // latch static field
            final Class caller = logCaller ? ClassLoaderResolver.getCallerClass (1) : null;
            
            StringBuffer buf = null;
            if ((caller != null) || (method != null))
            {
                buf = new StringBuffer ("[");
                
                if (caller != null) // if the caller could not be determined, s_classMask is ignored
                {
                    String callerName = caller.getName ();
                    
                    if (callerName.startsWith (PREFIX_TO_STRIP))
                        callerName = callerName.substring (PREFIX_TO_STRIP_LENGTH);
                        
                    String parentName = callerName;
                    
                    final int firstDollar = callerName.indexOf ('$');
                    if (firstDollar > 0) parentName = callerName.substring (0, firstDollar);
                    
                    if ((CLASS_MASK == null) || CLASS_MASK.contains (parentName))
                        buf.append (callerName);
                    else
                        return;
                }
                
                if (method != null)
                {
                    buf.append ("::");
                    buf.append (method);
                }
                
                buf.append ("] ");
            }
            
            if (PREFIX != null) out.print (PREFIX);
            if (buf != null)
            {
                if (msg != null) buf.append (msg);
                out.println (buf);
            }
            else
            {
                if (msg != null)
                    out.println (msg);
                else
                    out.println ();
            }
            
            if (FLUSH_LOG) out.flush ();
        }
    }
    
    private static void _log (final int level, final String method,
                              final String msg, final Throwable throwable)
    {        
        if ((level <= LEVEL) && (level >= SEVERE))
        {
            final PrintWriter out = OUT; // latch static field
            final Class caller = ClassLoaderResolver.getCallerClass (1);
            
            StringBuffer buf = null;
            if ((caller != null) || (method != null))
            {
                buf = new StringBuffer ("[");
                
                if (caller != null) // if the caller could not be determined, s_classMask is ignored
                {
                    String callerName = caller.getName ();
                    
                    if (callerName.startsWith (PREFIX_TO_STRIP))
                        callerName = callerName.substring (PREFIX_TO_STRIP_LENGTH);
                        
                    String parentName = callerName;
                    
                    final int firstDollar = callerName.indexOf ('$');
                    if (firstDollar > 0) parentName = callerName.substring (0, firstDollar);
                    
                    if ((CLASS_MASK == null) || CLASS_MASK.contains (parentName))
                        buf.append (callerName);
                    else
                        return;
                }
                
                if (method != null)
                {
                    buf.append ("::");
                    buf.append (method);
                }
                
                buf.append ("] ");
            }
            
            if (PREFIX != null) out.print (PREFIX);
            if (buf != null)
            {
                if (msg != null) buf.append (msg);
                out.println (buf);
            }
            else
            {
                if (msg != null)
                    out.println (msg);
                else
                    out.println ();
            }
            
            if (throwable != null)
            {
                throwable.printStackTrace (out);
            }
            
            if (FLUSH_LOG) out.flush ();
        }
    }

    
    
    // fields set based on LoggerInit configuration [LoggerInit must get
    // initialized before this class]:
    
    // NOTE: this design makes the API thread-safe without using method
    // synchronization; however, it also means the logging configuration
    // cannot be changed without reloading the relevant classes; this is
    // ok if coupled with implementing <top level class>.newInstance()
    // using a new classloader instance on each call.

    private static final int LEVEL          = LoggerInit.s_level;
    private static final PrintWriter OUT    = LoggerInit.s_out; // never null
    private static final Set CLASS_MASK     = LoggerInit.s_classMask;
    private static final String PREFIX      = LoggerInit.s_prefix; // never null
    
    private static final String PREFIX_TO_STRIP = "com.vladium."; // TODO: can this be set programmatically ?
    private static final int PREFIX_TO_STRIP_LENGTH = PREFIX_TO_STRIP.length ();
    private static final boolean FLUSH_LOG = true;
    
} // end of class
// ----------------------------------------------------------------------------